/** @file   car.cpp
 * @brief   Implementation of Car - class.
 * @version $Revision: 1.5 $
 * @date    $Date: 2006/08/13 10:50:22 $
 * @author  Tomi Lamminsaari
 */

#include "car.h"
#include "GfxManager.h"
#include <fstream>
#include <string>
#include "www_assert.h"
#include "www_map.h"
#include "carcontroller.h"
#include "settings.h"
#include "soundsamples.h"
#include "animplayer.h"
#include "warglobals.h"
#include "GfxId.h"
#include "AnimId.h"
#include "Resource/TextureBitmaps.rh"
using namespace eng2d;
using std::ifstream;
using std::string;

namespace WeWantWar {

/** Constructor.
 * @param     t                   Type of the car
 */
Car::Car( CarType t ) :
  PolygonObject( ),
  m_carType( t )
{
  // Create the polygons for this car
  int ret = this->prepareCar();
  WWW_ASSERT( ret == 0 );
  
  
  // Create the controller.
  this->setController( new CarController( this ) );
}



/** Constructs new Car and makes it drive the given route.
 */
Car::Car( CarType t, const string& routename ) :
  PolygonObject(),
  m_carType( t )
{
  // Create the polygons
  int ret = this->prepareCar();
  WWW_ASSERT( ret == 0 );
  
  // Create the controller.
  CarController* pC = new CarController( this );
  Route r = WarGlobals::pLevel->getRoute( routename );
  pC->setRoute( r );
  this->setController( pC );
}



/** Destructor.
 */
Car::~Car()
{
}


/** Updates the Car
 */
void Car::update()
{
  if ( this->state() == GameObject::STATE_KILLED ||
       this->state() == GameObject::STATE_DYING ) {
    return;
  }
  
  BaseController* pC = this->getController();
  pC->update();
  
  if ( pC->forward() == 2 ) {
    Vec2D mvec( m_movementVec );
    mvec.rotate( this->angle() );
    this->move( mvec );
    
  } else if ( pC->forward() == 1 ) {
    Vec2D mvec( m_movementVec );
    mvec /= 2.5;
    mvec.rotate( this->angle() );
    this->move( mvec );
  }
  this->changeAngle( pC->turn() * 2 );
  
  if ( pC->shoot() != 0 ) {
    this->makeSound( GameObject::SND_ATTACK );
    pC->m_shoot = 0;
  }
}



/** Kills the car
 */
void Car::kill()
{
  this->state( GameObject::STATE_KILLED );
  this->hidden( true );
                     
  Sound::playSample(SMP_GRENADE, false);
  const Animation& explosionAnim = GameAnims::findAnimation( AnimId::KExplosionGrenade );
  AnimPlayer::spawn( explosionAnim, this->position(), 0 );

  // And as the Car explodes, it emits some bullets
  float ac = 256.0 / 32.0;
  for (float a=0; a < 255; a+=ac) {
    eng2d::Vec2D spdVec( 0,-7 );
    spdVec.rotate( a );
    
    Bullet* b = BulletTable::createBullet( this, m_position,
                                           Bullet::ECarFragments );
    b->iRange = 110;
    b->iDamage = 10;
    b->setVelocity( spdVec );
    
    WarGlobals::pBulletManager->spawnBullet( b );
  }
  
}



/** Makes sound
 */
void Car::makeSound( GameObject::SoundID id ) const
{
  if ( id == GameObject::SND_PAIN ) {
    Sound::playSample( SMP_METALHIT, false );
    
  } else if ( id == GameObject::SND_ATTACK ) {
    Sound::playSample( SMP_CARHORN, false );
    
  }
}



/** Checks the bullet collisions
 */
bool Car::hitByBullet( Bullet* pB )
{
  this->causeDamage( pB );
  
  ParticleSparks* pP = new ParticleSparks( pB->iPosition, pB->velocity(), 12 );
  WarGlobals::pPartManager->addSystem( pP );
  return true;
}



/** Sets the route
 */
void Car::setRoute( const std::string& routeName )
{
  WaypointController* pC = dynamic_cast<WaypointController*>( this->getController() );
  pC->setRoute( WarGlobals::pLevel->getRoute( routeName ) );
}



//********************************************************************
//                                                                   *
//      Public GET - methods                                         *
//                                                                   *
//********************************************************************

/** Tells if Car is reloading
 */
bool Car::reloading() const
{
  return false;
}



/** Returns the type of this object.
 */
ObjectID::Type Car::objectType() const
{
  return ObjectID::TYPE_CAR;
}



//********************************************************************
//                                                                   *
//      Private methods                                              *
//                                                                   *
//********************************************************************

/** Creates the car polygons
 */
int Car::prepareCar()
{
  // Read the polygon data
  int ret = 0;
  string polyfile;
  if ( m_carType == CAR_TYPE1 ) {
    polyfile = "data/polygons/volvo_s80.txt";
  } else if ( m_carType == CAR_TYPE2 ) {
    polyfile = "data/polygons/volvo_xc70.txt";
  } else if ( m_carType == CAR_TYPE3 ) {
    polyfile = "data/polygons/volvo_s40.txt";
  } else if ( m_carType == CAR_TYPE4 ) {
    polyfile = "data/polygons/bmw_z4.txt";
  } else if ( m_carType == CAR_TYPE5 ) {
    polyfile = "data/polygons/toyota_rsc.txt";
  } else if ( m_carType == CAR_TYPE6 ) {
    polyfile = "data/polygons/ferrari.txt";
  } else if ( m_carType == CAR_TYPE_POLICE ) {
    polyfile = "data/polygons/policecar.txt";
  } else if ( m_carType == CAR_TYPE7 ) {
    polyfile = "data/polygons/taxi.txt";
  }

  ifstream fin( polyfile.c_str() );
  if ( !fin ) {
    return ret = -1;
    
  } else {
    string tmp;
    fin >> tmp;
    int ret = this->read( fin );
  
    fin.close();
  }

  // Add textures to this object
  GraphicsContainer* combine = GfxManager::findGfxContainer( GfxId::KTextureBitmaps );
  LOG_MESSAGE( "        Adding textures to car object" );
  if ( combine != 0 ) {
    this->addTexture( combine->GetBitmap(Textures_VolvoS80White) );
    this->addTexture( combine->GetBitmap(Textures_VolvoXC70Red) );
    this->addTexture( combine->GetBitmap(Textures_VolvoS40Yellow) );
    this->addTexture( combine->GetBitmap(Textures_BmwZ4White) );
    this->addTexture( combine->GetBitmap(Textures_ToyotaRscGrey) );
    this->addTexture( combine->GetBitmap(Textures_FerrariF40Red) );
    this->addTexture( combine->GetBitmap(Textures_PoliceCar1) );
    this->addTexture( combine->GetBitmap(Textures_Taxi1) );
  }
  LOG_MESSAGE( "        Textures added" );
  
  // Set the armor and other attributes.
  ObjectID::Type oid = ObjectID::TYPE_CAR;
  
  if ( m_carType == CAR_TYPE1 ) {
    m_movementVec.y( - Settings::floatObjProp(oid, "volvo_s80_spd:") );
    this->boundingSphere( Settings::floatObjProp(oid,"volvo_s80_bsphere:") );
    this->setArmor( Settings::floatObjProp(oid, "volvo_s80_armor:") );
    
  } else if ( m_carType == CAR_TYPE2 ) {
    m_movementVec.y( - Settings::floatObjProp(oid, "volvo_xc70_spd:") );
    this->boundingSphere( Settings::floatObjProp(oid, "volvo_xc70_bsphere:") );
    this->setArmor( Settings::floatObjProp(oid, "volvo_xc70_armor:") );
    
  } else if ( m_carType == CAR_TYPE3 ) {
    m_movementVec.y( - Settings::floatObjProp(oid, "volvo_s40_spd:") );
    this->boundingSphere( Settings::floatObjProp(oid, "volvo_s40_bsphere:") );
    this->setArmor( Settings::floatObjProp(oid, "volvo_s40_armor:") );
    
  } else if ( m_carType == CAR_TYPE4 ) {
    m_movementVec.y( - Settings::floatObjProp(oid, "bmw_z4_spd:") );
    this->boundingSphere( Settings::floatObjProp(oid, "bmw_z4_bsphere:") );
    this->setArmor( Settings::floatObjProp(oid, "bmw_z4_armor:") );
    
  } else if ( m_carType == CAR_TYPE5 ) {
    m_movementVec.y( - Settings::floatObjProp(oid, "toyota_rsc_spd:") );
    this->boundingSphere( Settings::floatObjProp(oid, "toyota_rsc_bsphere:") );
    this->setArmor( Settings::floatObjProp(oid, "toyota_rsc_armor:") );
    
  } else if ( m_carType == CAR_TYPE6 ) {
    m_movementVec.y( - Settings::floatObjProp(oid, "ferrari_spd:") );
    this->boundingSphere( Settings::floatObjProp(oid, "ferrari_bsphere:") );
    this->setArmor( Settings::floatObjProp(oid, "ferrari_armor:") );
    
  } else if ( m_carType == CAR_TYPE_POLICE ) {
    m_movementVec.y( - Settings::floatObjProp(oid, "police_spd:") );
    this->boundingSphere( Settings::floatObjProp(oid, "police_bsphere:") );
    this->setArmor( Settings::floatObjProp(oid, "police_armor:") );
    
  } else if ( m_carType == CAR_TYPE7 ) {
    m_movementVec.y( - Settings::floatObjProp(oid, "taxi_spd:") );
    this->boundingSphere( Settings::floatObjProp(oid, "taxi_bsphere:") );
    this->setArmor( Settings::floatObjProp(oid, "taxi_armor:") );
  }
  
  // Set the 4 collisionpoints
  
  return ret;
}


}
